<?php
namespace Auxin\Plugin\Pro\Elementor\Elements\Theme_Builder;

use Elementor\Plugin;
use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Image_Size;
use Elementor\Utils;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Css_Filter;

if ( ! defined( 'ABSPATH' ) ) {
  exit; // Exit if accessed directly.
}

/**
 * Elementor 'Post_Featured_Image' widget.
 *
 * Elementor widget that displays an 'Post_Featured_Image'.
 *
 * @since 1.0.0
 */
class Post_Featured_Image extends Widget_Base {

    /**
     * Get widget name.
     *
     * Retrieve 'Post_Featured_Image' widget name.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name() {
        return 'aux_featured_image';
    }

    /**
     * Get widget title.
     *
     * Retrieve 'Post_Featured_Image' widget title.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title() {
        return __('Featured Image', PLUGIN_DOMAIN );
    }

    /**
     * Get widget icon.
     *
     * Retrieve 'Post_Featured_Image' widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon() {
        return 'eicon-featured-image auxin-badge';
    }

    /**
     * Get widget categories.
     *
     * Retrieve 'Post_Featured_Image' widget icon.
     *
     * @since 1.0.0
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_categories() {
        return array( 'auxin-core', 'auxin-theme-elements-single' );
    }

    /**
     * Register 'Post_Featured_Image' widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     * @access protected
     */
    protected function register_controls() {

		$this->start_controls_section(
			'section_image',
			[
				'label' => __( 'Image', PLUGIN_DOMAIN ),
			]
		);

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            array(
                'name'       => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
                'separator'  => 'none',
                'default'    => 'large'
            )
        );

		$this->add_responsive_control(
			'align',
			[
				'label' => __( 'Alignment', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', PLUGIN_DOMAIN ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', PLUGIN_DOMAIN ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', PLUGIN_DOMAIN ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}}' => 'text-align: {{VALUE}};',
				],
			]
		);

        $this->add_control(
            'preloadable',
            array(
                'label'        => __('Preload image',PLUGIN_DOMAIN ),
                'type'         => Controls_Manager::SWITCHER,
                'label_on'     => __( 'On', PLUGIN_DOMAIN ),
                'label_off'    => __( 'Off', PLUGIN_DOMAIN ),
                'return_value' => 'yes',
                'default'      => 'no'
            )
        );

        $this->add_control(
            'preload_preview',
            array(
                'label'        => __('While loading image display',PLUGIN_DOMAIN ),
                'label_block'  => true,
                'type'         => Controls_Manager::SELECT,
                'options'      => auxin_get_preloadable_previews(),
                'return_value' => 'yes',
                'default'      => 'yes',
                'condition'    => array(
                    'preloadable' => 'yes'
                )
            )
        );

        $this->add_control(
            'preload_bgcolor',
            array(
                'label'     => __( 'Placeholder color while loading image', PLUGIN_DOMAIN ),
                'type'      => Controls_Manager::COLOR,
                'condition' => array(
                    'preloadable'     => 'yes',
                    'preload_preview' => array('no', 'simple-spinner', 'simple-spinner-light', 'simple-spinner-dark')
                )
            )
        );

		$this->add_control(
			'link_to',
			[
				'label' => __( 'Link', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::SELECT,
				'default' => 'none',
				'options' => [
					'none' => __( 'None', PLUGIN_DOMAIN ),
					'file' => __( 'Media File', PLUGIN_DOMAIN ),
					'custom' => __( 'Custom URL', PLUGIN_DOMAIN ),
				],
			]
		);

		$this->add_control(
			'link',
			[
				'label' => __( 'Link', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true
				],
				'placeholder' => __( 'https://your-link.com', PLUGIN_DOMAIN ),
				'condition' => [
					'link_to' => 'custom'
				],
				'show_label' => false
			]
		);

		$this->add_control(
			'view',
			[
				'label' => __( 'View', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::HIDDEN,
				'default' => 'traditional',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image',
			[
				'label' => __( 'Image', PLUGIN_DOMAIN ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'width',
			[
				'label' => __( 'Width', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ '%', 'px', 'vw' ],
				'range' => [
					'%' => [
						'min' => 1,
						'max' => 100,
					],
					'px' => [
						'min' => 1,
						'max' => 1000,
					],
					'vw' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .aux-media-image img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'space',
			[
				'label' => __( 'Max Width', PLUGIN_DOMAIN ) . ' (%)',
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'unit' => '%',
				],
				'tablet_default' => [
					'unit' => '%',
				],
				'mobile_default' => [
					'unit' => '%',
				],
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 1,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .aux-media-image img' => 'max-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'separator_panel_style',
			[
				'type' => Controls_Manager::DIVIDER,
				'style' => 'thick',
			]
		);

		$this->start_controls_tabs( 'image_effects' );

		$this->start_controls_tab( 'normal',
			[
				'label' => __( 'Normal', PLUGIN_DOMAIN ),
			]
		);

		$this->add_control(
			'opacity',
			[
				'label' => __( 'Opacity', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .aux-media-image img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters',
				'selector' => '{{WRAPPER}} .aux-media-image img',
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab( 'hover',
			[
				'label' => __( 'Hover', PLUGIN_DOMAIN ),
			]
		);

		$this->add_control(
			'opacity_hover',
			[
				'label' => __( 'Opacity', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 1,
						'min' => 0.10,
						'step' => 0.01,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .aux-media-image:hover img' => 'opacity: {{SIZE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Css_Filter::get_type(),
			[
				'name' => 'css_filters_hover',
				'selector' => '{{WRAPPER}} .aux-media-image:hover img',
			]
		);

		$this->add_control(
			'background_hover_transition',
			[
				'label' => __( 'Transition Duration', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'max' => 3,
						'step' => 0.1,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .aux-media-image img' => 'transition-duration: {{SIZE}}s',
				],
			]
		);

		$this->add_control(
			'hover_animation',
			[
				'label' => __( 'Hover Animation', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::HOVER_ANIMATION,
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'image_border',
				'selector' => '{{WRAPPER}} .aux-media-frame',
				'separator' => 'before',
			]
		);

		$this->add_responsive_control(
			'image_border_radius',
			[
				'label' => __( 'Border Radius', PLUGIN_DOMAIN ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .aux-media-frame' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'image_box_shadow',
				'exclude' => [
					'box_shadow_position',
				],
				'selector' => '{{WRAPPER}} .aux-media-frame',
			]
		);

		$this->end_controls_section();

    }

  /**
   * Render image box widget output on the frontend.
   *
   * Written in PHP and used to generate the final HTML.
   *
   * @since 1.0.0
   * @access protected
   */
  protected function render() {
    $settings = $this->get_settings_for_display();
    global $post;

    $media_size = empty( $settings['image_size'] ) ? 'medium_large' : $settings['image_size'];

    if( 'custom' == $media_size ){
        $media_size = array( 'width' => $settings['image_custom_dimension']['width'], 'height' => $settings['image_custom_dimension']['height'] );
    }

    $the_attach = auxin_get_the_post_responsive_thumbnail(
        $post->ID,
        array(
            'size'            => $media_size,
            'crop'            => true,
            'add_hw'          => true,
            'preloadable'     => $settings['preloadable'],
            'preload_preview' => $settings['preload_preview'],
            'preload_bgcolor' => $settings['preload_bgcolor'],
            'image_sizes'     => 'auto',
            'srcset_sizes'    => 'auto',
            'add_image_hw'    => false, // whether add width and height attr or not
            'upscale'         => true
        )
    );

    if( ! $the_attach ) {
        $this->add_render_attribute( 'image-src', 'src', Utils::get_placeholder_image_src() );
        $the_attach = sprintf( '<img %s />', $this->get_render_attribute_string( 'image-src' ) );
    }

    $this->add_render_attribute( 'wrapper', 'class', 'aux-media-frame aux-media-image' );

    if ( $settings['link_to'] !== 'none' ) {
		$url = $settings['link_to'] === 'file' ?  get_permalink() : $settings['link'];
        $this->add_render_attribute( 'url', 'href', $url );
        $the_attach = sprintf( '<a %1$s>%2$s</a>', $this->get_render_attribute_string( 'url' ), $the_attach );
    }

    echo sprintf( '<div %1$s>%2$s</div>', $this->get_render_attribute_string( 'wrapper' ), $the_attach );

  }

}
